/*******************************************************************************
* Sistema..................: PPT_Solver
* Unit.....................: USimAnn                          Tipo.......: .cpp
* Autor....................: Geraldo Regis Mauri (KAPA)
* ltima atualizao.......: 10 de JANEIRO de 2003
* Objetivo.................: Criar estruturas referentes ao Simulated Annealing.
* Descrio................: Esta unit contm a "implementao" de uma classe
*                            (TSimAnn) referente ao "mtodo" Simulated Annealing.
* Obs......................: O arquivo .h dessa unit possui a declarao de
*                            tudo que foi implementado aqui.
* Dependncias.............:
*     1) UTripulacoes
*     2) UTar_Jorn
*     3) UFrmEscDiaria
*     4) UFrmEscMensal
*******************************************************************************/

#pragma hdrstop

#include "USimAnn.h"

//------------------------------------------------------------------------------
#pragma package(smart_init)
//------------------------------------------------------------------------------

// ------------------------- Arquivos "importados" -------------------------- \\
#include "stdio.h"
#include "vcl.h"
#include <Math.h>        // exp
#include <limits.h>      // INT_MAX
#include <dateutils.hpp> // IncMinute


#include "UFrmEscDiaria.h"
#include "UFrmEscMensal.h"
#include "UTripulacoes.h"
#include "UTar_Jorn.h"
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: TSimAnn
- Objetivo...: Criar e inicializar o objeto TSimAnn.
- Retorno....: <-- Nenhum -->
- Parmetros.: <-- Nenhum -->
============================================================================= */
TSimAnn::TSimAnn()
{
 NumMaxIte   = 0;
 TxResf      = 0;
 TprInic     = 0;
 TprCong     = 0;
 TmpMaxProc  = 0;
 CPtprCong   = false;
 CPtmMaxProc = false;

 TempoExec     = 0;
 TempAtual     = 0;
 Status        = 0;
 PodeReaquecer = true;

 this->LerPadrao();
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: PreencherAtributos
- Objetivo...: Preencher os valores do objeto com os valores recebidos.
- Retorno....: <-- Nenhum -->
- Parmetros.:
    1) NMI   -> Nmero mximo de iteraes.
    2) TR    -> Taxa de resfriamento.
    3) TI    -> Temperatura inicial.
    4) TC    -> Temperatura de congelamento.
    5) TMP   -> Tempo mximo de processamento.
    6) CPTC  -> Critrio de parada por temperatura de congelamento.
    7) CPTMP -> Critrio de parada por tempo mximo de processamento.
============================================================================= */
void TSimAnn::PreencherAtributos(int NMI, double TR, double TI, double TC, int TMP,
                                 bool CPTC, bool CPTMP)
{
 NumMaxIte   = NMI;
 TxResf      = TR;
 TprInic     = TI;
 TprCong     = TC;
 TmpMaxProc  = TMP;
 CPtprCong   = CPTC;
 CPtmMaxProc = CPTMP;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: ValoresValidos
- Objetivo...: Verificar os valores do objeto.
- Retorno....: true/false (bool)
- Parmetros.: <-- Nenhum -->
============================================================================= */
bool TSimAnn::ValoresValidos()
{
 if ( (this->NumMaxIte > 0) && (this->TxResf > 0) && (this->TxResf < 1) &&
      (this->TprInic > 0) && (this->TprCong > 0) && (this->TmpMaxProc > 0) &&
      (this->CPtprCong || this->CPtmMaxProc) )
   return true;
 else
   return false;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: EstadoInicial
- Objetivo...: Setar os valores da temperatura e do tempo de execuo para o
               incio da execuo do SA.
- Retorno....: <-- Nenhum -->
- Parmetros.: <-- Nenhum -->
============================================================================= */
void TSimAnn::EstadoInicial()
{
 TempAtual     = TprInic;
 TempoExec     = 0;
 PodeReaquecer = true;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: Reaquecer
- Objetivo...: Reaquecer o sistema.
- Retorno....: <-- Nenhum -->
- Parmetros.:
   1) Temp -> Novo temperatura inicial.
============================================================================= */
void TSimAnn::Reaquecer(double Temp)
{
 TempAtual = Temp;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: CriterioDeParada
- Objetivo...: Verificar se o critrio de parada escolhido foi atendido.
- Retorno....: Parar ou no (false/true);
- Parmetros.:
   1) Time -> Tempo de encerramento.
   2) Temp -> Temperatura atual.
============================================================================= */
bool TSimAnn::CriterioDeParada(TDateTime Time, double Temp)
{
 if ( (CPtmMaxProc) && (CPtprCong) )
  {
   return ( (Time > TDateTime::CurrentDateTime()) && (Temp > TprCong) );
  }
 else if (CPtmMaxProc)
  {
   if (PodeReaquecer)
    return ( (Time > TDateTime::CurrentDateTime()) && (Temp > TprCong) );
   else
    return (Time > TDateTime::CurrentDateTime());
  }
 else // if (CPtprCong)
  {
   return (Temp > TprCong);
  }
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: EscolherTripRem
- Objetivo...: Selecionar uma tripulao aleatria que possua pelo menos uma tarefa.
- Retorno....: Nmero da tripulao (int).
- Parmetros.:
   1) E -> Objeto que representa uma escala diria.
============================================================================= */
int TSimAnn::EscolherTripRem(TEscDiaria *E)
{
 TTripED *Trip;
 int TripRem;

 do
  {
   TripRem = random(E->LstTripulacoes->Count);
   Trip = (TTripED *)E->LstTripulacoes->Items[TripRem];
  }while(Trip->LstTarefas->Count <= 0);

 return TripRem;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: EscolherTripAdic
- Objetivo...: Selecionar uma tripulao aleatria diferente da nmero TripRem.
- Retorno....: Nmero da tripulao (int).
- Parmetros.:
   1) E       -> Objeto que representa uma escala diria.
   2) TripRem -> Tripulao de onde ser removida a tarefa.
============================================================================= */
int TSimAnn::EscolherTripAdic(TEscDiaria *E, int TripRem)
{
 int TripAdic;

 do
  {
   TripAdic = random(E->LstTripulacoes->Count);
  }while(TripAdic == TripRem);

 return TripAdic;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: EscolherTarefaRem
- Objetivo...: Selecionar (aleatriamente) uma tarefa da tripulao nmero TripRem.
- Retorno....: Nmero da tarefa (int).
- Parmetros.:
   1) E       -> Objeto que representa uma escala diria.
   2) TripRem -> Tripulao de onde ser removida a tarefa.
============================================================================= */
int TSimAnn::EscolherTarefaRem(TEscDiaria *E, int TripRem)
{
 TTripED *Trip;
 int PosRem;

 Trip = (TTripED *)E->LstTripulacoes->Items[TripRem];
 PosRem = random(Trip->LstTarefas->Count);

 return PosRem;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: ExecutarED
- Objetivo...: Executar o mtodo Simulated Annealing sobre a escala diria.
- Retorno....: <-- Nenhum -->
- Parmetros.:
   1) ParED -> Objeto que contm os parmetros para a escala diria.
   2) PesED -> Objeto que contm os pesos para a escala diria.
============================================================================= */
void TSimAnn::ExecutarED(TParPptED *ParED, TPesosED *PesED)
{
 TTripED *TripAuxRem, *TripAuxAdic;
 TTarefa *TarAux;
 TEscDiaria *E;
 int TripRem, TripAdic, PosRem, PosAdic;

 int Iter = 0;
 Extended X = 0;
 long int FOTotalMelhor = 0;
 long int FOTotalAntes  = 0;
 long int FOTotalDepois = 0;

 Status = 0;

 Extended Temp = TempAtual;
 TempoExec = TDateTime::CurrentDateTime() - TempoExec;

 TDateTime TempoFim = IncMinute(TempoExec, TmpMaxProc);

 escdiaria->EfetuarCalculos(ParED, PesED);
 FOTotalMelhor = escdiaria->FncObjTotal;
 FrmEscDiaria->LblFOSC->Caption = IntToStr(FOTotalMelhor);
 FrmEscDiaria->LblFOMS->Caption = IntToStr(FOTotalMelhor);
 E = escdiaria->Clone();
 FOTotalAntes = FOTotalMelhor;

 // Enquanto o critrio de parada escolhido no for satisfeito
 while(CriterioDeParada(TempoFim, Temp))
  {
   FrmEscDiaria->LblFOSC->Caption = IntToStr(FOTotalAntes);

   // Enquanto o nmero de iteraes for menor que o mximo
   // Enquanto o sistema no atingir o equilbrio trmico
   while(Iter < NumMaxIte)
    {
     Application->ProcessMessages();
     if (Status == 1) //estado de interrupo
      {
       E->Free();
       TempoExec = TDateTime::CurrentDateTime() - TempoExec;
       this->TempAtual = Temp;
       FrmEscDiaria->LblTExec->Caption  = TempoExec.TimeString();
       FrmEscDiaria->LblTAtual->Caption = FloatToStrF(Temp, 15, 5, 4);
       return;
      }

     Iter++;

     // Escolher as tripulaes e a tarefa para o movimento.
     TripRem     = EscolherTripRem(E);
     TripAuxRem  = (TTripED *)E->LstTripulacoes->Items[TripRem];
     TripAdic    = EscolherTripAdic(E, TripRem);
     TripAuxAdic = (TTripED *)E->LstTripulacoes->Items[TripAdic];
     PosRem      = EscolherTarefaRem(E, TripRem);

     // Guardar o endereo da tarefa a ser movida.
     TarAux = (TTarefa *)TripAuxRem->LstTarefas->Items[PosRem];

     FOTotalAntes = E->FncObjTotal;

     // Fazer o movimento
     E->Movimento(&TripAuxRem->LstTarefas, PosRem, &TripAuxAdic->LstTarefas);

     // Efetuar os clculos depois do movimento.
     E->EfCalcDepoisDoMovimento(ParED, PesED, TripRem, TripAdic);
     FOTotalDepois = E->FncObjTotal;

     // TESTES DE ACEITAO DO MOVIMENTO \\
     if(FOTotalDepois < FOTotalAntes) // Se houver melhora na funo objetivo
      {
       FOTotalAntes = FOTotalDepois; // Movimento aceito

       // Se a funo objetivo for melhor que a "MELHOR" funo objetivo
       if(FOTotalAntes < FOTotalMelhor)
        {
         FOTotalMelhor = FOTotalAntes;
         FrmEscDiaria->LblFOMS->Caption = IntToStr(FOTotalMelhor);
         escdiaria->Free();
         escdiaria = E->Clone();
        }
      }//if
     else // Se NO houver melhora na funo objetivo
      {
       X = random(INT_MAX);
       X = X/INT_MAX;
       if(X < expl(-(FOTotalDepois - FOTotalAntes)/Temp))
        FOTotalAntes = FOTotalDepois; // Movimento aceito
       else
        {
         // Desfazer o movimento
         PosAdic = TripAuxAdic->LstTarefas->IndexOf(TarAux);
         E->Movimento(&TripAuxAdic->LstTarefas, PosAdic, &TripAuxRem->LstTarefas);

         // Efetuar os clculos depois de desfazer o movimento.
         E->EfCalcDepoisDoMovimento(ParED, PesED, TripRem, TripAdic);
        }//else
      }//else
    }//while(Iter < NumMaxIte)
   Temp = TxResf*Temp;
   Iter = 0;
  }//while(CriterioDeParada(TempoFim,Temp))

 E->Free();
 this->TempAtual = Temp;
 TempoExec = TDateTime::CurrentDateTime() - TempoExec;
 FrmEscDiaria->LblTExec->Caption  = TempoExec.TimeString();
 FrmEscDiaria->LblTAtual->Caption = FloatToStrF(Temp, 15, 5, 4);
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: EscolherTrip1Troca
- Objetivo...: Selecionar uma tripulao aleatria para trocar uma jornada.
- Retorno....: Nmero da tripulao (int).
- Parmetros.:
   1) EM -> Objeto que representa uma escala mensal.
============================================================================= */
int TSimAnn::EscolherTrip1Troca(TEscMensal *EM)
{
 return random(EM->LstTripulacoes->Count);
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: EscolherTrip2Troca
- Objetivo...: Selecionar uma tripulao diferente da Trip1.
- Retorno....: Nmero da tripulao (int).
- Parmetros.:
   1) EM    -> Objeto que representa uma escala mensal.
   2) Trip1 -> Uma das tripulaes que participaro da troca.
============================================================================= */
int TSimAnn::EscolherTrip2Troca(TEscMensal *EM, int Trip1)
{
 int Trip2;

 do
  {
   Trip2 = random(EM->LstTripulacoes->Count);
  }while(Trip2 == Trip1);

 return Trip2;
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: EscolherJornTroca
- Objetivo...: Selecionar (aleatriamente) uma jornada  da tripulao Trip2.
- Retorno....: Nmero da jornada (int).
- Parmetros.:
   1) EM    -> Objeto que representa uma escala mensal.
   2) Trip1 -> Uma das tripulaes que participaro da troca.
   3) Trip2 -> Outra tripulao que participar da troca.
============================================================================= */
int TSimAnn::EscolherJornTroca(TEscMensal *EM, int Trip1, int Trip2)
{
 TTripEM *TM1, *TM2;
 TDias *D1, *D2;
 int PosT;

 do
  {
   TM1 = (TTripEM *)EM->LstTripulacoes->Items[Trip1];
   TM2 = (TTripEM *)EM->LstTripulacoes->Items[Trip2];
   PosT = random(TM1->LstDias->Count);
   D1 = (TDias *)TM1->LstDias->Items[PosT];
   D2 = (TDias *)TM2->LstDias->Items[PosT];
  }while( (D1->Numero == -1) || (D2->Numero == -1) );

 return PosT;
}
//------------------------------------------------------------------------------


/* =============================================================================
- Mtodo.....: ExecutarEM
- Objetivo...: Executar o mtodo Simulated Annealing sobre a escala mensal.
- Retorno....: <-- Nenhum -->
- Parmetros.:
   1) EstM  -> Objeto que representa a estrutura do ms selecionado.
   2) ParEM -> Objeto que contm os parmetros para a escala mensal.
   3) PesEM -> Objeto que contm os pesos para a escala mensal.
============================================================================= */
void TSimAnn::ExecutarEM(TEstruturaMes *EstM, TParPptEM *ParEM, TPesosEM *PesEM)
{
 TTripEM *TripAux1, *TripAux2;
 TEscMensal *E;
 int Trip1, Trip2, PosTroca;

 int Iter = 0;
 Extended X = 0;
 long int FOTotalMelhor = 0;
 long int FOTotalAntes = 0;
 long int FOTotalDepois = 0;

 Status = 0;

 Extended Temp = TempAtual;
 TempoExec = TDateTime::CurrentDateTime() - TempoExec;

 TDateTime TempoFim = IncMinute(TempoExec, TmpMaxProc);

 escmensal->EfetuarCalculos(EstM, ParEM, PesEM);
 FOTotalMelhor = escmensal->FncObjTotal;
 FrmEscMensal->LblFOSC->Caption = IntToStr(FOTotalMelhor);
 FrmEscMensal->LblFOMS->Caption = IntToStr(FOTotalMelhor);
 E = escmensal->Clone();
 FOTotalAntes = FOTotalMelhor;

 // Enquanto o critrio de parada escolhido no for satisfeito
 while(CriterioDeParada(TempoFim, Temp))
  {
   FrmEscMensal->LblFOSC->Caption = IntToStr(FOTotalAntes);

   // Enquanto o nmero de iteraes for menor que o mximo
   // Enquanto o sistema no atingir o equilbrio trmico
   while(Iter < NumMaxIte)
    {
     Application->ProcessMessages();
     if (Status == 1) //estado de concluso
      {
       E->Free();
       TempoExec = TDateTime::CurrentDateTime() - TempoExec;
       TempAtual = Temp;
       FrmEscMensal->LblTExec->Caption  = TempoExec.TimeString();
       FrmEscMensal->LblTAtual->Caption = FloatToStrF(Temp, 15, 5, 4);
       return;
      }

     Iter++;

     // Escolher as tripulaes e a tarefa para o movimento.
     Trip1 = EscolherTrip1Troca(E);
     TripAux1 = (TTripEM *)E->LstTripulacoes->Items[Trip1];
     Trip2 = EscolherTrip2Troca(E, Trip1);
     TripAux2 = (TTripEM *)E->LstTripulacoes->Items[Trip2];
     PosTroca = EscolherJornTroca(E, Trip1, Trip2);

     FOTotalAntes = E->FncObjTotal;

     // Fazer o movimento
     E->Movimento(&TripAux1->LstDias, PosTroca, &TripAux2->LstDias);

     // Efetuar os clculos depois do movimento.
     E->EfCalcDepoisDoMovimento(EstM, ParEM, PesEM, Trip1, Trip2);
     FOTotalDepois = E->FncObjTotal;

     // TESTES DE ACEITAO DO MOVIMENTO \\
     if(FOTotalDepois < FOTotalAntes) // Se houver melhora na funo objetivo
      {
       FOTotalAntes = FOTotalDepois; // Movimento aceito

       // Se a funo objetivo for melhor que a "MELHOR" funo objetivo
       if(FOTotalAntes < FOTotalMelhor)
        {
         FOTotalMelhor = FOTotalAntes;
         FrmEscMensal->LblFOMS->Caption = IntToStr(FOTotalMelhor);
         escmensal->Free();
         escmensal = E->Clone();
        }
      }//if
     else // Se NO houver melhora na funo objetivo
      {
       X = random(INT_MAX);
       X = X/INT_MAX;
       if(X < exp(-(FOTotalDepois - FOTotalAntes)/Temp))
        FOTotalAntes = FOTotalDepois; // Movimento aceito
       else
        {
         // Desfazer o movimento
         E->Movimento(&TripAux2->LstDias, PosTroca, &TripAux1->LstDias);

         // Efetuar os clculos depois de desfazer o movimento.
         E->EfCalcDepoisDoMovimento(EstM, ParEM, PesEM, Trip1, Trip2);
        }//else
      }//else
    }//while(Iter < NumMaxIte)
   Temp = TxResf*Temp;
   Iter = 0;
  }//while(CriterioDeParada(TempoFim,Temp))

 E->Free();
 this->TempAtual = Temp;
 TempoExec = TDateTime::CurrentDateTime() - TempoExec;
 FrmEscMensal->LblTExec->Caption  = TempoExec.TimeString();
 FrmEscMensal->LblTAtual->Caption = FloatToStrF(Temp, 15, 5, 4);
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: SelecionarArquivo
- Objetivo...: Ler os valores do objeto a partir do arquivo correto.
- Retorno....: <-- Nenhum -->
- Parmetros.: Tipo -> 'D'- Default   |  'P'- Padro.
============================================================================= */
void TSimAnn::SelecionarArquivo(char Tipo)
{
 String Diretorio, Msg;
 if (Tipo == 'P')
  {
   Diretorio = ExtractFilePath(Application->ExeName) + "\\Arquivos\\SimAnn.par";
   Msg = "O arquivo 'SimAnn.par' no pode ser aberto!";
  }
 else
  {
   Diretorio = ExtractFilePath(Application->ExeName) + "\\Arquivos\\PPT.dft";
   Msg = "O arquivo 'PPT.dft' no pode ser aberto!";
  }

 char *Path = new char[ Diretorio.Length() + 1 ];
 strcpy(Path, Diretorio.c_str());

 FILE *Arq;
 Arq = fopen(Path, "r");
 delete(Path);
 if(!Arq)
  {
   Beep();
   MessageDlg(Msg, mtError, TMsgDlgButtons() << mbOK, 0);
  }
 else
  {
   char P[100];
   char *Par = "SAN";

   int NI;
   int TM;
   char TR[15];
   char TI[15];
   char TC[15];
   char CTM[5];
   char CTC[5];

   do
    {
     fscanf(Arq, "%s", &P);
    }while (strncmp(Par, P, 4));

   fscanf(Arq, "%d\t%s\t%s\t%s\t%d\t%s\t%s", &NI,&TR,&TI,&TC,&TM,&CTC,&CTM);

   this->NumMaxIte   = NI;
   this->TmpMaxProc  = TM;
   this->TxResf      = StrToFloat(TR);
   this->TprInic     = StrToFloat(TI);
   this->TprCong     = StrToFloat(TC);
   this->CPtmMaxProc = StrToBool(CTM);
   this->CPtprCong   = StrToBool(CTC);
  }

 fclose(Arq);
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: LerPadrao
- Objetivo...: Ler os valores do objeto a partir do arquivo SimAnn.par.
- Retorno....: <-- Nenhum -->
- Parmetros.: <-- Nenhum -->
============================================================================= */
void TSimAnn::LerPadrao()
{
 SelecionarArquivo('P');
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: LerPadrao
- Objetivo...: Ler os valores do objeto a partir do arquivo PPT.dft.
- Retorno....: <-- Nenhum -->
- Parmetros.: <-- Nenhum -->
============================================================================= */
void TSimAnn::LerDefault()
{
 SelecionarArquivo('D');
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: GravarPadrao
- Objetivo...: Gravar os valores do objeto no arquivo SimAnn.par.
- Retorno....: <-- Nenhum -->
- Parmetros.: <-- Nenhum -->
============================================================================= */
void TSimAnn::GravarPadrao()
{
 String Diretorio = ExtractFilePath(Application->ExeName) + "\\Arquivos\\SimAnn.par";

 char *Path = new char[ Diretorio.Length() + 1 ];
 strcpy(Path, Diretorio.c_str());

 FILE *Arq;
 Arq = fopen(Path, "w");
 delete(Path);
 if(!Arq)
  {
   Beep();
   MessageDlg("O arquivo 'SimAnn.par' no pode ser aberto!", mtError,
              TMsgDlgButtons() << mbOK, 0);
  }
 else
  {
   String CTC = "false";
   String CTM = "false";

   if (this->CPtprCong)
    CTC = "true";
   if (this->CPtmMaxProc)
    CTM = "true";

   fprintf(Arq, "%s\t%d\t%s\t%s\t%s\t%d\t%s\t%s", "SAN", this->NumMaxIte,
           FloatToStr(this->TxResf), FloatToStr(this->TprInic), FloatToStr(this->TprCong),
           this->TmpMaxProc, CTC, CTM );
  }

 fclose(Arq);
}
//------------------------------------------------------------------------------

/* =============================================================================
- Mtodo.....: Free
- Objetivo...: Destuir o objeto TSimAnn.
- Retorno....: <-- Nenhum -->
- Parmetros.: <-- Nenhum -->
============================================================================= */
void TSimAnn::Free()
{
 delete(this);
}
//------------------------------------------------------------------------------

// ----------------------------------- FIM ---------------------------------- \\
